<?php

namespace App\Http\Controllers;

use App\Models\Categories;
use Illuminate\Http\Request;
use Carbon\Carbon;

class CategoriesController extends Controller
{
    public function index()
    {
        // Obter todas as categorias do banco de dados
        $categories = Categories::orderBy('updated_at', 'desc')->paginate(6);

        // Formatar a data de last_update
        foreach ($categories as $category) {
            $category->last_update = Carbon::parse($category->updated_at)->format('d/m/Y');
        }

        // Retornar a view com as categorias para exibição
        return view('dashboard.categories', ['categories' => $categories]);
    }

    public function create()
    {
        return view('dashboard.categories.create');
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|max:255',
        ]);

        if (! Categories::where('name', $validatedData['name'])->doesntExist()) {
            return redirect()->back()->with('error', 'Já existe uma categoria com este nome!');
        }

        Categories::create($validatedData);

        return redirect()->back()->with('success', 'Categoria criada com sucesso!');
    }



    public function edit(Categories $category)
    {
        return view('dashboard.categories.edit', compact('category'));
    }

    public function update(Request $request, Categories $category)
    {
        // Validar os dados do formulário
        $validatedData = $request->validate([
            'name' => 'required|max:255',
        ]);

        // Verificar se já existe uma categoria com o nome atualizado
        if (Categories::where('name', $validatedData['name'])->where('id', '!=', $category->id)->exists()) {
            // Se a categoria já existe, retorne uma mensagem de erro
            return redirect()->back()->with('error', 'Já existe uma categoria com este nome!');
        }

        // Atualizar a categoria com os dados validados
        $category->update($validatedData);

        // Redirecionar de volta à página de categorias com uma mensagem de sucesso
        return redirect()->back()->with('success', 'Categoria atualizada com sucesso!');
    }

    public function destroy(Categories $category)
    {
        $category->delete();
        return redirect()->back()->with('success', 'Categoria excluida com sucesso!');
    }
}
