<?php

namespace App\Http\Controllers;

use App\Http\Requests\NewsStoreRequest;
use App\Http\Requests\NewsEditRequest;
use App\Models\Categories;
use App\Models\News;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\File;


class NewsController extends Controller
{
    public function index()
    {
        // Obter todas as categorias do banco de dados
        $posts = News::orderBy('updated_at', 'desc')->paginate(4);

        // Formatar a data de last_update
        foreach ($posts as $post) {
            $post->last_update = Carbon::parse($post->updated_at)->format('d/m/Y');
        }

        // Retornar a view com as categorias para exibição
        return view('dashboard.posts', ['posts' => $posts]);
    }

    public function create()
    {
        $categories = Categories::pluck('name', 'id')->toArray();
        return view('dashboard.new-post', compact('categories'));
    }


    public function store(NewsStoreRequest $request)
    {
        // Obtenha os dados validados da solicitação
        $validatedData = $request->validated();

        // Verifique se uma imagem de capa foi fornecida
        if ($request->hasFile('capa')) {
            // Faça o upload da nova imagem de capa
            $requestImage = $request->file('capa');
            $extension = $requestImage->extension();
            $imageName = md5($requestImage->getClientOriginalName() . strtotime("now")) . "." . $extension;
            $requestImage->move(public_path('img/uploads'), $imageName);

            // Adicione o nome da imagem de capa aos dados validados
            $validatedData['capa'] = $imageName;
        }

        // Remover o campo 'slug' do array validado para evitar duplicatas
        unset($validatedData['slug']);

        // Gerar automaticamente o slug a partir do título
        $slug = Str::slug($validatedData['title']);
        $originalSlug = $slug;
        $count = 1;

        // Verificar se já existe um post com o mesmo slug
        while (News::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . str_pad($count, 2, '0', STR_PAD_LEFT);
            $count++;
        }

        // Adicionar o slug gerado aos dados validados
        $validatedData['slug'] = $slug;

        // Criar o post com os dados validados e o slug gerado
        News::create($validatedData);

        // Redirecione para a página de índice dos posts
        return to_route('posts')->with('success', 'Post criado com sucesso!');
    }


    public function edit(News $post)
    {
        $categories = Categories::pluck('name', 'id')->toArray();
        return view('dashboard.edit-post', compact('post', 'categories'));
    }

    public function update(News $post, NewsEditRequest $request)
    {
        // Obtenha os dados validados da solicitação
        $requestData = $request->validated();

        // Verifique se uma nova imagem de capa foi enviada
        if ($request->hasFile('capa')) {
            // Obtenha o nome da imagem atual do post
            $currentImageName = $post->capa;

            // Faça o upload da nova imagem de capa
            $requestImage = $request->file('capa');
            $extension = $requestImage->extension();
            $imageName = md5($requestImage->getClientOriginalName() . strtotime("now")) . "." . $extension;
            $requestImage->move(public_path('img/uploads'), $imageName);

            // Atualize o nome da imagem no campo 'capa' dos dados do post
            $requestData['capa'] = $imageName;

            // Verifique se o nome da imagem atual é diferente do nome da nova imagem
            if ($currentImageName !== $imageName) {
                // Exclua a imagem atual
                File::delete(public_path('img/uploads/' . $currentImageName));
            }
        } else {
            // Se não houver uma nova imagem de capa, remova a validação do campo 'capa'
            unset($requestData['capa']);
        }

        // Atualizar os dados do post
        $post->update($requestData);

        // Redirecione para a página de índice dos posts
        return to_route('posts')->with('success', 'Post atualizado com sucesso!');
    }


    public function destroy(News $post)
    {
        $post->delete();
        return redirect()->back()->with('success', 'Post excluido com sucesso!');
    }
}
